/*
  Antenna Rotator Turn Indicator
  CDE/HyGain version

  Glen Popiel - KW5GP

  Uses the AdaFruit NeoPixel Library

  Released under the GPLv3 license
*/

//#define debug // Uncomment this to enable the display of debug information on the Serial Monitor

#include <Adafruit_NeoPixel.h>  // Include the Adafruit NeoPixel Library

#define led_pin 6 // Define the Neopixel I/O pin

#define num_pixels 12 // NeoPixel Strip/Ring size

#define brake_pin 2 //define the brake input pin
#define ccw_pin 3 //define the counterclockwise rotation input pin
#define cw_pin 4 // define the clockwise rotation input pin
#define debounce 20 // Set a 20ms debounce period

// Setup the NeoPixel library
// Uncomment the following for standard RGB Neopixel/WS281x LED Strips
//Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRB + NEO_KHZ800);

// Uncomment the following for RGBW Neopixel/WS281x LED Strips (RGB w/White LED)
// If you look at the unlit LEDs, half of each LED is covered with yellow
Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRBW + NEO_KHZ800);

#define delay_time 500 // Time (in milliseconds) to pause between pixels

int current_pixel, last_pixel; // Pixel variables
bool cw_in, ccw_in, brake_in, moving; // Rotation variables

void setup()
{
#ifdef debug  // Start the serial port only if debugging is enabled
  Serial.begin(9600);
#endif
  pinMode(cw_pin, INPUT_PULLUP);  // Set the clockwise pin mode
  pinMode(ccw_pin, INPUT_PULLUP);  // Set the counterclockwise pin mode
  pinMode(brake_pin, INPUT_PULLUP);  // Set the brake pin mode

  pixels.begin(); // Initialize the NeoPixel display
  pixels.clear(); // Set all pixel colors to 'off'
  pixels.show();   // Send the updated pixel colors to the hardware
  moving = false; // Clear the moving flag
}

void loop()
{
  brake_in = digitalRead(brake_pin);  // Check to see if the brake release switch is pressed
  delay(debounce);
  if (brake_in == LOW)  // If the brake release is pressed
  {
#ifdef debug
    Serial.println("Brake enabled");
#endif

    brake_in = digitalRead(brake_pin);  // Read the brake release switch
    cw_in = digitalRead(cw_pin);  // Read the clockwise (right) switch
    ccw_in = digitalRead(ccw_pin);  // Read the counterclockwise (left) switch
    delay(debounce);

    if ((brake_in == HIGH && cw_in == HIGH && ccw_in == HIGH) || ((brake_in == LOW) && (cw_in == HIGH && ccw_in == HIGH)))
    {
      // A switch transition delay caused invalid condition
      moving = false; // Turn off the rotating flag
    } else {

      if ((cw_in == HIGH || ccw_in == HIGH))  // One or both direction switches is pressed
      {
#ifdef debug
        Serial.print( "Brake = ");
        Serial.print(brake_in);
        Serial.print("   CCW = ");
        Serial.print(ccw_in);
        Serial.print("   CW = ");
        Serial.println(cw_in);
        Serial.println("CW or CCW Switch pressed");
#endif

        moving = true;  // Set the rotating flag
        if (cw_in == HIGH)
        {
          // we're turning clockwise

#ifdef debug
          Serial.println("Turning Clockwise");
#endif
          turn_cw();  // Call the turn_cw() function
        }

        if (ccw_in == HIGH)
        {
          // We're turning counterclockwise

#ifdef debug
          Serial.println("Turning CounterClockwise");
#endif

          turn_ccw();  // Call the turn_ccw() function
        }
      } else
      {
        // Have we been moving? If so, flash the brake lights
        if (moving)
        {
          moving = false; // turn off the rotating flag
        }
        // N more rotation - clear the LED's
        pixels.clear(); // Set all pixel colors to 'off'
        pixels.show();  // Send the updated pixel colors to the hardware
      }
    }
  }
}

void turn_cw()  // The turn_cw() function
{
  int current_pixel = 0;
  delay(debounce);
  // Repeat the loop while the cw (right) switch is pressed
  while (digitalRead(cw_pin) == HIGH && digitalRead(brake_pin) == LOW)
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0));    // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel >= num_pixels)  // Check to see if we've exceeding the number of pixels in the ring
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware
      current_pixel = current_pixel - num_pixels; // Reset the pixel counter
    } else
    {
      current_pixel = current_pixel + 1;  // increment the pixel to light
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function when the cw switch is released
}


void turn_ccw()  // The turn_ccw() function
{
  int current_pixel = num_pixels - 1; // Start at the top and light the pixels counterclockwise
  delay(debounce);
  // Repeat the loop while the ccw (left) switch is pressed
  while (digitalRead(ccw_pin) == HIGH  && digitalRead(brake_pin) == LOW)
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0));    // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel < 0)  // Check to see if we've exceeding the number of pixels in the ring
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware

      current_pixel = num_pixels - 1; // decrement the pixel to light
    } else
    {
      current_pixel = current_pixel - 1;  // turn on the next LED
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function when the ccw switch is released
}

void brake()  // The brake() function
{
#ifdef debug
  Serial.println("Braking");
#endif

  pixels.fill(0, 0, num_pixels); // Turn off all the LEDs
  pixels.show();   // Send the updated pixel colors to the hardware.
  delay(delay_time);
  for (int i = 0; i < 5; i = i + 1)  // Flash all LEDs for 5 seconds for braking indicator
  {
    pixels.fill(pixels.Color(150, 0, 0), 0, num_pixels);  // Light all LEDs with Red
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
    pixels.fill(0, 0, num_pixels); // Turn off all the LEDs
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
  }
}

